<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Package;
use App\Models\Server;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;

class PackageController extends Controller
{
    /**
     * Display a listing of the packages.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $packages = Package::with('servers')
            ->orderBy('sort_order')
            ->get();
        
        return view('admin.packages.index', compact('packages'));
    }

    /**
     * Show the form for creating a new package.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $types = ['rank', 'money', 'custom_lua', 'console_command', 'combo'];
        $actionTypes = ['rank', 'money', 'custom_lua', 'console_command', 'pointshop', 'pointshop2', 'pointshop2_premium'];
        $servers = Server::where('is_active', true)->orderBy('name')->get();
        
        return view('admin.packages.create', compact('types', 'actionTypes', 'servers'));
    }

    /**
     * Store a newly created package in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'required|string',
            'price' => 'required|numeric|min:0',
            'type' => 'required|string|in:rank,money,custom_lua,console_command,combo',
            'features' => 'nullable|array',
            'features.*' => 'required|string|max:255',
            'actions_type' => 'nullable|array',
            'actions_type.*' => 'required_with:actions_value|string|in:rank,money,custom_lua,console_command',
            'actions_value' => 'nullable|array',
            'actions_value.*' => 'required_with:actions_type|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'is_active' => 'boolean',
            'sort_order' => 'required|integer|min:0',
            'servers' => 'nullable|array',
            'servers.*' => 'exists:servers,id',
            // Stock fields
            'stock_enabled' => 'boolean',
            'stock_quantity' => 'nullable|required_if:stock_enabled,1|integer|min:1',
            // Subscription fields
            'is_subscription' => 'boolean',
            'subscription_duration' => 'nullable|required_if:is_subscription,1|integer|min:1',
        ]);
        
        $validated['slug'] = Str::slug($request->name);
        $validated['is_active'] = $request->has('is_active');
        $validated['stock_enabled'] = $request->has('stock_enabled');
        $validated['is_subscription'] = $request->has('is_subscription');
        
        // Set stock_remaining equal to stock_quantity if stock is enabled
        if ($validated['stock_enabled'] && isset($validated['stock_quantity'])) {
            $validated['stock_remaining'] = $validated['stock_quantity'];
        }
        
        // Filter out empty features
        if (isset($validated['features'])) {
            $validated['features'] = array_filter($validated['features'], function($feature) {
                return !empty(trim($feature));
            });
            $validated['features'] = array_values($validated['features']);
        }
        
        // Process actions data if present
        if (isset($request->actions_type) && isset($request->actions_value)) {
            $actionsData = [];
            foreach ($request->actions_type as $key => $type) {
                if (!empty($type) && isset($request->actions_value[$key]) && !empty($request->actions_value[$key])) {
                    $actionsData[] = [
                        'type' => $type,
                        'value' => $request->actions_value[$key]
                    ];
                }
            }
            $validated['actions'] = $actionsData;
        }
        
        // Handle image upload
        if ($request->hasFile('image')) {
            $image = $request->file('image');
            $filename = time() . '.' . $image->getClientOriginalExtension();
            $path = $image->storeAs('packages', $filename, 'public');
            $validated['image_path'] = $path;
        }
        
        // Remove servers from validated data before creating package
        $serverIds = $validated['servers'] ?? [];
        unset($validated['servers']);
        unset($validated['actions_type']);
        unset($validated['actions_value']);
        
        $package = Package::create($validated);
        
        // Attach servers to package
        if (!empty($serverIds)) {
            $package->servers()->attach($serverIds);
        }
        
        return redirect()->route('admin.packages.index')
            ->with('success', 'Package created successfully.');
    }

    /**
     * Display the specified package.
     *
     * @param  \App\Models\Package  $package
     * @return \Illuminate\Http\Response
     */
    public function show(Package $package)
    {
        $package->load('servers', 'transactions');
        return view('admin.packages.show', compact('package'));
    }

    /**
     * Show the form for editing the specified package.
     *
     * @param  \App\Models\Package  $package
     * @return \Illuminate\Http\Response
     */
    public function edit(Package $package)
    {
        $types = ['rank', 'money', 'custom_lua', 'console_command', 'combo'];
        $actionTypes = ['rank', 'money', 'custom_lua', 'console_command'];
        $servers = Server::where('is_active', true)->orderBy('name')->get();
        $assignedServers = $package->servers->pluck('id')->toArray();
        
        return view('admin.packages.edit', compact('package', 'types', 'actionTypes', 'servers', 'assignedServers'));
    }

    /**
     * Update the specified package in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Package  $package
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Package $package)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'required|string',
            'price' => 'required|numeric|min:0',
            'type' => 'required|string|in:rank,money,custom_lua,console_command,combo',
            'features' => 'nullable|array',
            'features.*' => 'required|string|max:255',
            'actions_type' => 'nullable|array',
            'actions_type.*' => 'required_with:actions_value|string|in:rank,money,custom_lua,console_command',
            'actions_value' => 'nullable|array',
            'actions_value.*' => 'required_with:actions_type|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'is_active' => 'boolean',
            'sort_order' => 'required|integer|min:0',
            'servers' => 'nullable|array',
            'servers.*' => 'exists:servers,id',
            // Stock fields
            'stock_enabled' => 'boolean',
            'stock_quantity' => 'nullable|required_if:stock_enabled,1|integer|min:1',
            'stock_remaining' => 'nullable|integer|min:0',
            // Subscription fields
            'is_subscription' => 'boolean',
            'subscription_duration' => 'nullable|required_if:is_subscription,1|integer|min:1',
        ]);
        
        $validated['slug'] = Str::slug($request->name);
        $validated['is_active'] = $request->has('is_active');
        $validated['stock_enabled'] = $request->has('stock_enabled');
        $validated['is_subscription'] = $request->has('is_subscription');
        
        // Handle stock updates
        if ($validated['stock_enabled']) {
            // If stock was just enabled, set remaining equal to quantity
            if (!$package->stock_enabled && isset($validated['stock_quantity'])) {
                $validated['stock_remaining'] = $validated['stock_quantity'];
            }
            // If stock quantity changed, adjust remaining proportionally
            elseif ($package->stock_enabled && isset($validated['stock_quantity']) && $validated['stock_quantity'] != $package->stock_quantity) {
                // Admin manually set remaining stock
                if (isset($validated['stock_remaining'])) {
                    // Ensure remaining doesn't exceed new quantity
                    $validated['stock_remaining'] = min($validated['stock_remaining'], $validated['stock_quantity']);
                } else {
                    // Keep existing remaining
                    $validated['stock_remaining'] = min($package->stock_remaining, $validated['stock_quantity']);
                }
            }
        } else {
            // Stock disabled, clear stock fields
            $validated['stock_quantity'] = null;
            $validated['stock_remaining'] = null;
        }
        
        // Filter out empty features
        if (isset($validated['features'])) {
            $validated['features'] = array_filter($validated['features'], function($feature) {
                return !empty(trim($feature));
            });
            $validated['features'] = array_values($validated['features']);
        }
        
        // Process actions data if present
        if (isset($request->actions_type) && isset($request->actions_value)) {
            $actionsData = [];
            foreach ($request->actions_type as $key => $type) {
                if (!empty($type) && isset($request->actions_value[$key]) && !empty($request->actions_value[$key])) {
                    $actionsData[] = [
                        'type' => $type,
                        'value' => $request->actions_value[$key]
                    ];
                }
            }
            $validated['actions'] = $actionsData;
        }
        
        // Handle image upload
        if ($request->hasFile('image')) {
            // Delete old image if exists
            if ($package->image_path) {
                Storage::disk('public')->delete($package->image_path);
            }
            
            $image = $request->file('image');
            $filename = time() . '.' . $image->getClientOriginalExtension();
            $path = $image->storeAs('packages', $filename, 'public');
            $validated['image_path'] = $path;
        }
        
        // Remove servers from validated data before updating package
        $serverIds = $validated['servers'] ?? [];
        unset($validated['servers']);
        unset($validated['actions_type']);
        unset($validated['actions_value']);
        
        $package->update($validated);
        
        // Sync servers with package
        $package->servers()->sync($serverIds);
        
        return redirect()->route('admin.packages.index')
            ->with('success', 'Package updated successfully.');
    }

    /**
     * Remove the specified package from storage.
     *
     * @param  \App\Models\Package  $package
     * @return \Illuminate\Http\Response
     */
    public function destroy(Package $package)
    {
        // Check if package has transactions
        if ($package->transactions()->count() > 0) {
            return redirect()->route('admin.packages.index')
                ->with('error', 'Cannot delete package with transactions.');
        }
        
        // Delete image if exists
        if ($package->image_path) {
            Storage::disk('public')->delete($package->image_path);
        }
        
        // Detach all servers
        $package->servers()->detach();
        
        $package->delete();
        
        return redirect()->route('admin.packages.index')
            ->with('success', 'Package deleted successfully.');
    }

    /**
     * Show package server assignments management
     *
     * @return \Illuminate\Http\Response
     */
    public function servers()
    {
        $packages = Package::with('servers')->orderBy('sort_order')->get();
        $servers = Server::where('is_active', true)->orderBy('name')->get();
        
        return view('admin.packages.servers', compact('packages', 'servers'));
    }

    /**
     * Update package server assignments in bulk
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function updateServers(Request $request)
    {
        $validated = $request->validate([
            'package_servers' => 'nullable|array',
            'package_servers.*' => 'array',
            'package_servers.*.*' => 'exists:servers,id',
        ]);

        $packageServers = $validated['package_servers'] ?? [];

        // Update each package's server assignments
        foreach ($packageServers as $packageId => $serverIds) {
            $package = Package::find($packageId);
            if ($package) {
                $package->servers()->sync($serverIds);
            }
        }

        return redirect()->route('admin.packages.servers')
            ->with('success', 'Package server assignments updated successfully.');
    }
}